/**
* \file: WaylandContext.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Android Auto
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2013 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#ifndef AAUTO_WAYLANDCONTEXT_H
#define AAUTO_WAYLANDCONTEXT_H

#include <atomic>
#include <list>
#include <aauto_macros.h>
#include <aauto/util/shared_ptr.h>
#include <aauto/util/Mutex.h>
#include <pthread_adit.h>
#include <wayland-util.h>
#include <wayland-client-protocol.h>
#include <compositor-shim.h>
#include "aauto/AditInputSource.h"

namespace adit { namespace aauto
{
class WaylandContext;

class WlSeat
{
public:
    WlSeat()
    :mSeatName(nullptr)
    ,mSeatId(0)
    ,mWlSeat(nullptr)
    ,mWlTouch(nullptr)
    ,mWlPointer(nullptr)
    ,mWlKeyboard(nullptr)
    ,mParentWaylandContext(nullptr)
    {
    }
    ~WlSeat()
    {
        if (mSeatName != nullptr) {
            free (mSeatName);
            mSeatName = nullptr;
        }
        aauto_safe_call(wl_seat_destroy, mWlSeat);
        aauto_safe_call(wl_touch_release, mWlTouch);
        aauto_safe_call(wl_pointer_release, mWlPointer);
        aauto_safe_call(wl_keyboard_release, mWlKeyboard);
    }

    char*               GetSeatName()         { return mSeatName; }
    uint32_t            GetSeatId()           { return mSeatId; }
    struct wl_seat*     GetWLSeat()     const { return mWlSeat; }
    struct wl_touch*    GetWLTouch()    const { return mWlTouch; }
    struct wl_pointer*  GetWLPointer()  const { return mWlPointer; }
    struct wl_keyboard* GetWLKeyboard() const { return mWlKeyboard; }

    WaylandContext* GetParentWaylandContext() { return mParentWaylandContext; }

    void SetSeatName(const char* inSeatName) { mSeatName = strdup(inSeatName); }
    void SetSeatId(uint32_t inSeatId) { mSeatId = inSeatId; }
    void SetWLSeat(struct wl_seat* inWlSeat) { mWlSeat = inWlSeat; }
    void SetWLTouch(struct wl_touch* inWlTouch) { mWlTouch = inWlTouch; }
    void SetWLPointer(struct wl_pointer* inWlPointer) { mWlPointer = inWlPointer; }
    void SetWLKeyboard(struct wl_keyboard* inWlKeyboard) { mWlKeyboard = inWlKeyboard; }

    void SetParentWaylandContext(WaylandContext* pWlContext) { mParentWaylandContext = pWlContext; }
private:
    char*               mSeatName;
    uint32_t            mSeatId;
    struct wl_seat*     mWlSeat;
    struct wl_touch*    mWlTouch;
    struct wl_pointer*  mWlPointer;
    struct wl_keyboard* mWlKeyboard;

    class WaylandContext* mParentWaylandContext;
};

class WaylandContext
{
public:
    /* abstract base for register-able device listeners */
    class DeviceListener
    {
    public:
        virtual ~DeviceListener();
    };

    WaylandContext(IAditInputSourceCallbacks* inCallbacks);
    WaylandContext(struct wl_display* inDisplay, IAditInputSourceCallbacks* inCallbacks);
    ~WaylandContext();

    bool Initialize();

    inline struct wl_display* GetDisplay() { return display; }
    inline struct wl_compositor* GetCompositor() { return compositor; }
    inline struct compositor_shim_context* GetAdapterContext() { return adapterContext; }
    inline struct wl_shm* GetShm() { return shm; }
    inline uint32_t GetShmFormat() { return shmFormat; }

    void SetTouchListener(struct wl_touch_listener* inListener,
            shared_ptr<DeviceListener> inObj);
    void SetPointerListener(struct wl_pointer_listener* inListener,
            shared_ptr<DeviceListener> inObj);

    bool StartInputThread(uint32_t inProcessIntervalUSec);
    void StopInputThread();

    void AddWlSeat(WlSeat* inSeat) {
        Autolock l(&mLockSeatList);
        mSeatList.push_back(inSeat);
    }
    std::list<WlSeat*>& getWlSeatList(void) { return mSeatList; }
    Mutex mLockSeatList;
private:

    /* list of binded wl_seat's */
    std::list<WlSeat*> mSeatList;

    struct wl_display* display;
    bool   shareWlDisplay;

    struct wl_event_queue* inputQueue;
    struct wl_compositor* compositor;
    struct wl_shm* shm;
    uint32_t shmFormat;
    struct wl_registry* registry;

    struct compositor_shim_context* adapterContext;

    struct wl_touch_listener* touchListener;
    shared_ptr<DeviceListener> touchListenerObj;

    struct wl_pointer_listener* pointerListener;
    shared_ptr<DeviceListener> pointerListenerObj;

    std::atomic<bool> threadRunning;
    pthread_t threadId;
    uint32_t processIntervalUSec;
    int32_t shutDownEventFd;
    const uint64_t shutDownEvent = 1;

    IAditInputSourceCallbacks* mCallbacks;

    static void* inputThread(void* inData);

    /* static listener information */
    static struct wl_registry_listener registryListener;
    static struct wl_shm_listener shmListener;
    static struct wl_seat_listener seatListener;
    static bool staticInitialized;

    static void staticInitialize();

    static void onRegistryGlobal(void* inMe, struct wl_registry* inRegistry, uint32_t inName,
            const char* inInterface, uint32_t inVersion);
    static void onRegistryGlobalRemove(void* inMe, struct wl_registry* inRegistry,
            uint32_t inName);
    static void onShmFormats(void* inData, struct wl_shm* inShm,
            uint32_t inFormat);
    static void onSeatCapabilities(void* inData, struct wl_seat* inSeat, uint32_t inCaps);
    static void onSeatHandleGetName(void* inData, struct wl_seat* inSeat, const char *name);

    void destroyWaylandContext();
};

} } /* namespace adit { namespace aauto */

#endif /* AAUTO_WAYLANDCONTEXT_H */
